#!/bin/bash
#
#  File: test-nested-xserver.sh
#
#  Copyright 2024-2025 Penguin Solutions Inc.  All Rights Reserved.
#

set +e

# Localhost address the client/browser can connect to server
CLIENT_IP=127.0.0.1

# Defaults:
HOST_DISPLAY_ID=:1
GUEST_DISPLAY_ID=:16

CLIENT_IRW_PORT=8891

WM_TITLE="Xephyr ${USER}"
XSERVER="Xvfb"

START_GNOME_SHELL=0

GNOME_SHELL_OPTIONS=--x11
GNOME_SESSION_OPTIONS=

START_CLIENT=1

VERBOSE=1

MINIMIZE_WINDOW=NOT-SET

function vecho() {
    if [[ ${VERBOSE} -ge $1 ]]; then
        echo "$2"
    fi
}

function verror() {
    bg_red=`tput setab 1`
    white=`tput setaf 15`
    reset=`tput sgr0`
    echo
    echo ${bg_red}${white}"ERROR: $1"${reset}
    exit_script
}

function help() {
    echo
    echo "Usage: test-nested-xserver.sh [-h] [-x <executable>] [-G <display id>] [-H <display id>] [-p <irw port>] [-verbose <level>] [-gs] [-o <options>] [-n]"
    echo
    echo "test-nested-xserver.sh launches a nested X11 server session and starts a private irw session"
    echo "without any authentification."
    echo
    echo "  Options"
    echo "   -h, --help                     this help"
    echo "   -x, --xserver <executable>     window manager in /usr/bin, default 'Xvfb'"
    echo "   -G, --guestid <display id>     display id of guest x11 session, e.g. ':16'"
    echo "   -H, --hostid <display id>      display id of x11 session on host"
    echo "   -p, -cp, --port <port number>  irw port on client"
    echo "   -o, --opts <options>           options to window manager, can be empty to clear default options"
    echo "   -v, --verbose <level>          sets verbosity"
    echo "   -gs, --gnome-shell             use 'gnome-shell' as window manager (default 'gnome-session')"
    echo "   -o, --opts <options>           options to window manager, can be empty to clear default options"
    echo "   -n, --no-client                don't start client at end of script"
    echo "   -m, --minimize-window          minimize guest x11 window"
    echo
    echo "i.e. ./test-nested-xserver.sh -x Xephyr  -G :18  -H :1  -p 8400"
    echo "     ./test-nested-xserver.sh -x Xvfb  -G :18  -v 2  -p 8400"
}

print_params() {
    echo
    echo "Parameters:"
    echo "XSERVER, -x                = ${XSERVER}"
    echo "GUEST_DISPLAY_ID, -G       = ${GUEST_DISPLAY_ID}"
    echo "HOST_DISPLAY_ID, -H        = ${HOST_DISPLAY_ID}"
    echo "CLIENT_IRW_PORT, -p        = ${CLIENT_IRW_PORT}"
    echo "START_GNOME_SHELL, -gs     = ${START_GNOME_SHELL}"
    echo "GNOME_SESSION_OPTIONS, -o  = ${GNOME_SESSION_OPTIONS}"
    echo "GNOME_SHELL_OPTIONS, -o    = ${GNOME_SHELL_OPTIONS}"
    echo "START_CLIENT, -n           = ${START_CLIENT}"
    echo "VERBOSE, -v                = ${VERBOSE}"
    echo "MINIMIZE_WINDOW, -m        = ${MINIMIZE_WINDOW}"
    echo
}

check_params() {
    if [[ ${CLIENT_IRW_PORT} == "8890" ]]; then
        vecho 1
        vecho 1 "Using default port '${CLIENT_IRW_PORT}' to connect to irw server"
        vecho 1 "Please modify this value for each client connecting to the server"
    fi
    if [[ ${GUEST_DISPLAY_ID} == ":16" ]]; then
        vecho 1
        vecho 1 "Using default display id '${GUEST_DISPLAY_ID}' to start guest X server"
        vecho 1 "Please modify this value for each client connecting to the server"
    fi
    if [[ ${HOST_DISPLAY_ID} == ":1" ]]; then
        vecho 1
        vecho 1 "Using default display id '${HOST_DISPLAY_ID}' to identify the host xserver."
        vecho 1 "If you are not able to run this script, check if you need to modify this value"
    fi
    if [[ ${MINIMIZE_WINDOW_SET} -eq 0 ]]; then
        if [[ ${XSERVER} == "Xephyr" ]]; then
            MINIMIZE_WINDOW=1
        else
            MINIMIZE_WINDOW=0            
        fi
    fi  
}

trap exit_script INT
trap exit_script TERM

# Prevent recursive calls of exit_script 
exit_called=0

function exit_script() {

    if [ ${exit_called} -eq 1 ]; then
        exit 0
    else
        exit_called=1
    fi

    # Terminate X server using stored pid
    echo "Exiting..."
    kill -s SIGTERM ${x_pid}
    kill -s SIGTERM ${server_pid}

    exit 0
}

MINIMIZE_WINDOW_SET=0

while [[ $# -gt 0 ]]
do
    key="$1"
    case $key in
        -h|--help)
            help
            exit 0
            ;;
        -p|--port)
            CLIENT_IRW_PORT="$2"
            shift # past argument
            ;;
        -H|--hostid)
            HOST_DISPLAY_ID="$2"
            shift # past argument
            ;;
        -G|--guestid)
            GUEST_DISPLAY_ID="$2"
            shift # past argument
            ;;
        -gs|--gnome-shell)
            START_GNOME_SHELL=1;
            ;;
        -o|--opts)
            GNOME_SHELL_OPTIONS="$2";
            GNOME_SESSION_OPTIONS="$2";
            shift
            ;;
        -x|--xserver)
            XSERVER="$2"
            shift # past argument
            ;;
        -v|--verbose)
            VERBOSE="$2"
            shift
            ;;
        -n|--no-client)
            START_CLIENT=0
            ;;
        -m|--minimize-window)
            MINIMIZE_WINDOW_SET=1
            MINIMIZE_WINDOW=1
            ;;
        *)
            verror "unknown option: $key" 
        esac
    shift # past arg or value
done

check_params

if [[ ${VERBOSE} -gt 1 ]]; then
    print_params
fi

if [[ ! -f /opt/ice-remoteware/bin/ice-remoteware ]]; then
    verror "Can't find ICE RemoteWare server."
fi

if [[ ! -f /usr/bin/${XSERVER} ]]; then
    verror "Can't find X-server at /usr/bin/${XSERVER}"
fi

if [[ ${MINIMIZE_WINDOW} -eq 1 ]]; then
    if [[ ! -f /usr/bin/xdotool ]]; then
        vecho 1 "/usr/bin/xdotool not found, will not try to minimize the guest window"
        MINIMIZE_WINDOW=0
    fi
fi

vecho 1 "Launch X-server (${XSERVER}) ..."

if [[ ${XSERVER} == "Xephyr" ]]; then
    DISPLAY=${HOST_DISPLAY_ID} /usr/bin/Xephyr -br -ac -noreset -title "${WM_TITLE}" -screen 1920x1080 ${GUEST_DISPLAY_ID} -dpi 96 > /dev/null 2>&1 &
elif [[ ${XSERVER} == "Xvfb" ]]; then
    /usr/bin/Xvfb ${GUEST_DISPLAY_ID} & &> /dev/null
else
    /usr/bin/${XSERVER} ${GUEST_DISPLAY_ID} & &> /dev/null
fi

x_pid=$!

vecho 1 "... launched X-server (pid ${x_pid})"
vecho 1

sleep 1

if [[ ${MINIMIZE_WINDOW} -eq 1 ]]; then
    vecho 1 "Trying to minimize guest window"
    DISPLAY=${HOST_DISPLAY_ID} /usr/bin/xdotool search "${WM_TITLE}" windowminimize
    vecho 1 " ... minimized guest window"
fi

# When using gnome-session, to avoid auth dialog on server (rocky 9), see
# https://www.reddit.com/r/Ubuntu/comments/15stmwn/how_do_i_suppress_authentication_is_required_to/?rdt=42444

if [[ ${START_GNOME_SHELL} -eq 1 ]]; then
    WM_NAME=gnome-shell
    WM_OPTIONS=${GNOME_SHELL_OPTIONS}
else
    WM_NAME=gnome-session
    WM_OPTIONS=${GNOME_SESSION_OPTIONS}
fi

unset XDG_RUNTIME_DIR

vecho 1 "Launching ${WM_NAME} ..."

DISPLAY=${GUEST_DISPLAY_ID} dbus-run-session ${WM_NAME} ${WM_OPTIONS} > /dev/null 2>&1 &
gnome_pid=$!

sleep 2

ps -ef | awk '{print $2}' | grep ${gnome_pid}  > /dev/null

if [[ $? -ne 0 ]]; then
    vecho 1 "Error starting ${WM_NAME}, make sure the server is not running Wayland"
    vecho 1 "Please run: \"DISPLAY=${GUEST_DISPLAY_ID} dbus-run-session ${WM_NAME} ${WM_OPTIONS}\""
    vecho 1 "from the command line to diagnose this error."
    vecho 1 
    vecho 1 "X-server output will be black."
    vecho 1 
else
    vecho 1 "... ${WM_NAME} launched (pid ${gnome_pid})"
fi

vecho 2
vecho 2 "ice-remoteware options: --application  --setConfig=Server.X11.Display=${GUEST_DISPLAY_ID} --setConfig=Server.Network.Port=${CLIENT_IRW_PORT}"
vecho 2 "      --setConfig=Server.Network.Secure=false  --setConfig=Server.Log.FileName=~/ice-remoteware.log --setConfig=Server.Misc.LocalHostRestricted=false  --setConfig=Server.Auth.Enabled=false"

vecho 1
vecho 1 "Starting up ICE RemoteWare"
DISPLAY=${GUEST_DISPLAY_ID} /opt/ice-remoteware/bin/ice-remoteware \
    --application \
    --setConfig=Server.X11.Display=${GUEST_DISPLAY_ID} \
    --setConfig=Server.Network.Port=${CLIENT_IRW_PORT} \
    --setConfig=Server.Network.Secure=false \
    --setConfig=Server.Log.FileName=~/ice-remoteware.log \
    --setConfig=Server.Misc.LocalHostRestricted=false \
    --setConfig=Server.Auth.Enabled=false &> /dev/null &

server_pid=$!
vecho 1 "... started ICE RemoteWare (pid ${server_pid})"
vecho 1

vecho 0 "Server accessible at http://${CLIENT_IP}:${CLIENT_IRW_PORT}"

if [[ ${START_CLIENT} -eq 1 ]]; then
    if [[ $(uname) == "Darwin" ]]; then
        if [[ -d  "/Applications/ice-remoteware-client.app/" ]]; then
            open irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false
            if [[ $? -eq 0 ]]; then
                opened=1
            fi
        fi
        if [[ ${opened} -eq 0 ]]; then
            open http://${CLIENT_IP}:${CLIENT_IRW_PORT}
        fi
    else
        uname | grep -is CYGWIN > /dev/null
        if [[ $? -eq 0 ]]; then
            if [[ -d  "${PROGRAMFILES//\\//}/Penguin Solutions/ICE RemoteWare Client" ]]; then
                cmd /c start "" irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false
                if [[ $? -eq 0 ]]; then
                    opened=1
                fi
            fi
            if [[ ${opened} -eq 0 ]]; then
                cmd /c start "" http://${CLIENT_IP}:${CLIENT_IRW_PORT}
            fi
        else
            if [[ -d  "/opt/ice-remoteware-client/" ]]; then
                xdg-open irw://${CLIENT_IP}:${CLIENT_IRW_PORT}?secure=false > /dev/null 2>&1
                if [[ $? -eq 0 ]]; then
                    opened=1
                fi
            fi
            if [[ ${opened} -eq 0 ]]; then
                xdg-open http://${CLIENT_IP}:${CLIENT_IRW_PORT} > /dev/null 2>&1
            fi
        fi
    fi
else
    vecho 2 "Not starting connecting to server"
fi

vecho 1 "press ENTER or Ctrl-C to exit"
read

exit_script
